<?php

use App\Http\Controllers\DashboardController;
use App\Http\Controllers\ContainerController;
use App\Http\Controllers\BaplieController;
use App\Http\Controllers\DataController;
use App\Http\Controllers\BayplanController;
use Illuminate\Support\Facades\Route;

// Dashboard Routes
Route::get('/', [DashboardController::class, 'index'])->name('dashboard');

// BAPLIE Management Routes
Route::prefix('baplie')->name('baplie.')->group(function () {
    // BAPLIE management index
    Route::get('/management', [BaplieController::class, 'index'])->name('management');
    
    // ✅ Upload BAPLIE file (Main Upload - supports BAPLIE, Excel, or both)
    Route::post('/upload', [BaplieController::class, 'upload'])->name('upload');
    Route::get('/upload', function () {
        return redirect()->route('dashboard')
            ->with('info', 'Please use the upload form to upload files.');
    });
    
    // ✅ Upload Excel tambahan (untuk menambah/update data setelah BAPLIE ada)
    Route::post('/upload-additional-excel', [BaplieController::class, 'uploadAdditionalExcel'])
        ->name('upload-additional-excel');
    
    // ✅ Upload multiple Excel sekaligus (batch)
    Route::post('/upload-multiple-excel', [BaplieController::class, 'uploadMultipleExcel'])
        ->name('upload-multiple-excel');
    
    // Export EDI from session
    Route::get('/export-edi/{voyage?}', [BaplieController::class, 'exportEdi'])->name('export-edi');
    
    // Debug routes (development only)
    Route::get('/debug-weight', [BaplieController::class, 'debugWeight'])->name('debug.weight');
    
    // ✅ Container Operations (inside baplie prefix)
    Route::post('/containers/duplicate', [BaplieController::class, 'duplicateContainer'])
        ->name('containers.duplicate'); 
    
    Route::delete('/containers/{index}', [BaplieController::class, 'deleteContainer'])
        ->name('containers.delete');
    
    Route::put('/containers/{index}', [BaplieController::class, 'updateContainer'])
        ->name('containers.update');
    
    // Reports
    Route::get('/reefer-report', [BaplieController::class, 'reeferReport'])
        ->name('reefer.report');
    
    Route::get('/anomaly-report', [BaplieController::class, 'anomalyReport'])
        ->name('anomaly-report');
        
});

// ✅ Container Management Routes (standalone - untuk view detail individual)
Route::prefix('containers')->name('containers.')->group(function () {
    // List containers from session
    Route::get('/', [ContainerController::class, 'index'])->name('index');
    
    // Show specific container by ID
    Route::get('/{container}', [ContainerController::class, 'show'])->name('show');
    
    // Edit container by ID
    Route::get('/{container}/edit', [ContainerController::class, 'edit'])->name('edit');
    
    // Update container by ID (jika ada controller terpisah)
    Route::put('/{container}', [ContainerController::class, 'update'])->name('update');
    
    // Delete container by ID (jika ada controller terpisah)
    Route::delete('/{container}', [ContainerController::class, 'destroy'])->name('destroy');
});

// ✅ Bayplan Routes
Route::prefix('bayplan')->name('bayplan.')->group(function () {
    // Main bayplan view
    Route::get('/', [BaplieController::class, 'bayplan'])->name('index');
    
    // API endpoints untuk bayplan (jika diperlukan)
    Route::get('/api/containers', function() {
        $containers = session('containers', []);
        return response()->json($containers);
    })->name('api.containers');
});

// ✅ Data Management Routes
Route::prefix('data')->name('data.')->group(function () {
    // Clear all session data
    Route::delete('/clear', [DataController::class, 'destroy'])->name('delete');
    
    // Clear specific data types
    Route::delete('/clear/containers', [DataController::class, 'clearContainers'])
        ->name('clear.containers');
    
    Route::delete('/clear/voyage', [DataController::class, 'clearVoyage'])
        ->name('clear.voyage');
});

// ✅ API Routes (for AJAX requests)
Route::prefix('api')->name('api.')->group(function () {
    // Get container statistics
    Route::get('/stats/containers', [ContainerController::class, 'getStats'])
        ->name('stats.containers');
    
    // Get bayplan data
    Route::get('/bayplan/data', [BayplanController::class, 'getBayplanData'])
        ->name('bayplan.data');
    
    // Search containers
    Route::get('/containers/search', [ContainerController::class, 'search'])
        ->name('containers.search');
});

// ✅ Utility Routes
Route::prefix('utils')->name('utils.')->group(function () {
    // Download sample BAPLIE file
    Route::get('/sample-baplie', function () {
        $samplePath = storage_path('app/samples/sample_baplie.edi');
        
        if (file_exists($samplePath)) {
            return response()->download($samplePath);
        }
        
        return redirect()->back()->with('error', 'Sample file not found.');
    })->name('sample.baplie');
    
    // System health check
    Route::get('/health', function () {
        return response()->json([
            'status' => 'ok',
            'timestamp' => now()->toIso8601String(),
            'session' => [
                'has_containers' => session()->has('containers'),
                'containers_count' => count(session('containers', [])),
                'has_voyage' => session()->has('voyage'),
                'has_status_breakdown' => session()->has('status_breakdown'),
                'has_anomalies' => session()->has('container_anomalies'),
            ],
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
        ]);
    })->name('health');
});