<?php

namespace App\Http\Controllers\Transaction;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseDetail;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\Supplier;
use App\Models\Branch;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;
use BaconQrCode\Common\ErrorCorrectionLevel;
use BaconQrCode\Encoder\Encoder;

class PurchaseController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:purchases.index'])->only('index', 'show');
        $this->middleware(['permission:purchases.create'])->only(['create', 'save']);
        $this->middleware(['permission:purchases.edit'])->only(['edit', 'update']);
        $this->middleware(['permission:purchases.delete'])->only('delete');
    }

    /**
     * Display a listing of purchases
     */
    public function index()
    {
        $title = "purchase";
        $purchases = Purchase::with(['supplier', 'branch', 'user'])
            ->orderBy('id', 'desc')
            ->get();

        return view('transaction.purchase.index', compact('title', 'purchases'));
    }

    /**
     * Show the form for creating a new purchase
     */
    public function create()
    {
        $title = "purchase";
        $suppliers = Supplier::orderBy('id', 'desc')->get();
        $branches = Branch::orderBy('id', 'desc')->get();
        $products = Product::with(['brand', 'category'])->get();

        // Generate purchase number
        $purchaseNumber = $this->generatePurchaseNumber();

        return view('transaction.purchase.create', compact('title', 'suppliers', 'branches', 'products', 'purchaseNumber'));
    }

    /**
     * Store a newly created purchase
     */
    public function save(Request $request)
    {
        $this->validate($request, [
            'supplier_id' => 'required|exists:suppliers,id',
            'branch_id' => 'required|exists:branches,id',
            'purchase_date' => 'required|date',
            'details' => 'required|array|min:1',
            'details.*.product_id' => 'required|exists:products,id',
            'details.*.quantity' => 'required|numeric|min:1',
            'details.*.purchase_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            // Calculate total amount
            $totalAmount = 0;
            foreach ($request->details as $detail) {
                $totalAmount += $detail['quantity'] * $detail['purchase_price'];
            }

            // Create purchase
            $purchase = Purchase::create([
                'purchase_number' => $this->generatePurchaseNumber(),
                'barcode' => $this->generateBarcode(),
                'supplier_id' => $request->supplier_id,
                'branch_id' => $request->branch_id,
                'purchase_date' => $request->purchase_date,
                'total_amount' => $totalAmount,
                'status' => 'completed',
                'notes' => $request->notes,
                'user_id' => Auth::id(),
            ]);

            // Generate QR Code and save to public folder
            $this->generateQrCode($purchase);

            // Create purchase details and update stock
            foreach ($request->details as $detail) {
                // Create purchase detail
                PurchaseDetail::create([
                    'purchase_id' => $purchase->id,
                    'product_id' => $detail['product_id'],
                    'quantity' => $detail['quantity'],
                    'purchase_price' => $detail['purchase_price'],
                    'subtotal' => $detail['quantity'] * $detail['purchase_price'],
                ]);

                // Update or create product stock
                $productStock = ProductStock::where('product_id', $detail['product_id'])
                    ->where('branch_id', $request->branch_id)
                    ->first();

                if ($productStock) {
                    $productStock->increment('quantity', $detail['quantity']);
                } else {
                    ProductStock::create([
                        'product_id' => $detail['product_id'],
                        'branch_id' => $request->branch_id,
                        'quantity' => $detail['quantity'],
                    ]);
                }

                // Create stock movement record
                StockMovement::create([
                    'product_id' => $detail['product_id'],
                    'branch_id' => $request->branch_id,
                    'type' => 'in',
                    'quantity' => $detail['quantity'],
                    'reference_type' => 'App\Models\Purchase',
                    'reference_id' => $purchase->id,
                    'notes' => 'Pembelian #' . $purchase->purchase_number,
                    'user_id' => Auth::id(),
                ]);
            }

            DB::commit();

            return redirect()->route('purchase.show', $purchase->id)
                ->with(['success' => 'Data Pembelian Berhasil Disimpan! Anda dapat mencetak faktur pembelian.']);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with(['error' => 'Terjadi kesalahan: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified purchase
     */
    public function show(Purchase $purchase)
    {
        $title = "purchase";
        $purchase->load(['supplier', 'branch', 'user', 'purchaseDetails.product']);

        return view('transaction.purchase.show', compact('title', 'purchase'));
    }

    /**
     * Show the form for editing the specified purchase
     */
    public function edit(Purchase $purchase)
    {
        // Only allow editing if status is pending
        // if ($purchase->status !== 'pending') {
        //     return redirect()->route('purchase')
        //         ->with(['error' => 'Hanya pembelian dengan status pending yang dapat diubah!']);
        // }

        $title = "purchase";
        $suppliers = Supplier::orderBy('id', 'desc')->get();
        $branches = Branch::orderBy('id', 'desc')->get();
        $products = Product::with(['brand', 'category'])->get();
        $purchase->load(['purchaseDetails.product']);

        return view('transaction.purchase.edit', compact('title', 'purchase', 'suppliers', 'branches', 'products'));
    }

    /**
     * Update the specified purchase
     */
    public function update(Request $request, Purchase $purchase)
    {
        $this->validate($request, [
            'supplier_id' => 'required|exists:suppliers,id',
            'branch_id' => 'required|exists:branches,id',
            'purchase_date' => 'required',
            'details' => 'required|array|min:1',
            'details.*.product_id' => 'required|exists:products,id',
            'details.*.quantity' => 'required|numeric|min:1',
            'details.*.purchase_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();

        try {

            /* ===============================
            * 1. KEMBALIKAN STOK LAMA
            * =============================== */
            foreach ($purchase->purchaseDetails as $oldDetail) {

                $stock = ProductStock::where('product_id', $oldDetail->product_id)
                    ->where('branch_id', $purchase->branch_id)
                    ->first();

                if ($stock) {
                    $stock->decrement('quantity', $oldDetail->quantity);
                }

                // hapus movement lama
                StockMovement::where('reference_type', Purchase::class)
                    ->where('reference_id', $purchase->id)
                    ->where('product_id', $oldDetail->product_id)
                    ->delete();
            }

            /* ===============================
            * 2. HITUNG TOTAL BARU
            * =============================== */
            $totalAmount = 0;
            foreach ($request->details as $detail) {
                $totalAmount += $detail['quantity'] * $detail['purchase_price'];
            }

            /* ===============================
            * 3. UPDATE PURCHASE
            * =============================== */
            $purchase->update([
                'supplier_id' => $request->supplier_id,
                'branch_id' => $request->branch_id,
                'purchase_date' => $request->purchase_date,
                'total_amount' => $totalAmount,
                'notes' => $request->notes,
            ]);

            /* ===============================
            * 5. HAPUS DETAIL LAMA
            * =============================== */
            $purchase->purchaseDetails()->delete();

            /* ===============================
            * 6. SIMPAN DETAIL BARU + STOK
            * =============================== */
            foreach ($request->details as $detail) {

                PurchaseDetail::create([
                    'purchase_id' => $purchase->id,
                    'product_id' => $detail['product_id'],
                    'quantity' => $detail['quantity'],
                    'purchase_price' => $detail['purchase_price'],
                    'subtotal' => $detail['quantity'] * $detail['purchase_price'],
                ]);

                // update / create stock
                $stock = ProductStock::firstOrCreate(
                    [
                        'product_id' => $detail['product_id'],
                        'branch_id' => $request->branch_id,
                    ],
                    ['quantity' => 0]
                );

                $stock->increment('quantity', $detail['quantity']);

                // stock movement baru
                StockMovement::create([
                    'product_id' => $detail['product_id'],
                    'branch_id' => $request->branch_id,
                    'type' => 'in',
                    'quantity' => $detail['quantity'],
                    'reference_type' => Purchase::class,
                    'reference_id' => $purchase->id,
                    'notes' => 'Update Pembelian #' . $purchase->purchase_number,
                    'user_id' => Auth::id(),
                ]);
            }

            DB::commit();

            return redirect()->route('purchase.show', $purchase->id)
                ->with('success', 'Data Pembelian berhasil diperbarui');

        } catch (\Exception $e) {

            DB::rollBack();

            return redirect()->back()
                ->withInput()
                ->with('error', 'Gagal update pembelian: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified purchase
     */
    public function delete(Purchase $purchase)
    {
        // Only allow deleting if status is pending
        // if ($purchase->status !== 'pending') {
        //     return redirect()->route('purchase')
        //         ->with(['error' => 'Hanya pembelian dengan status pending yang dapat dihapus!']);
        // }

        DB::beginTransaction();
        try {
            // Delete purchase details
            $purchase->purchaseDetails()->delete();

            // Delete purchase
            $purchase->delete();

            DB::commit();

            return redirect()->route('purchase')
                ->with('success', 'Data Pembelian Berhasil Dihapus');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('purchase')
                ->with(['error' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }

    /**
     * Get product price for specific branch
     */
    public function getProductPrice(Request $request)
    {
        $productId = $request->input('product_id');
        $branchId = $request->input('branch_id');

        if (!$productId || !$branchId) {
            return response()->json([
                'success' => false,
                'message' => 'Product ID dan Branch ID harus diisi'
            ], 400);
        }

        // Get the latest price for the product at the specific branch
        $productPrice = \App\Models\ProductPrice::where('product_id', $productId)
            ->where('branch_id', $branchId)
            ->orderBy('effective_date', 'desc')
            ->first();

        if ($productPrice) {
            return response()->json([
                'success' => true,
                'purchase_price' => $productPrice->purchase_price,
                'selling_price' => $productPrice->selling_price,
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Harga belum diatur untuk produk ini di cabang yang dipilih'
        ], 404);
    }

    /**
     * Get products by branch (only products that have stock in the selected branch)
     */
    public function getProductsByBranch(Request $request)
    {
        $branchId = $request->input('branch_id');

        if (!$branchId) {
            // If no branch selected, return all products
            $products = Product::with(['brand', 'category'])->get();

            return response()->json([
                'success' => true,
                'products' => $products->map(function($product) {
                    return [
                        'id' => $product->id,
                        'name' => $product->name,
                        'unit' => $product->unit,
                        'brand_name' => $product->brand->name ?? '-',
                        'stock' => 0,
                    ];
                })
            ]);
        } else {
            // Get only products that have stock in the selected branch
            $productStocks = ProductStock::where('branch_id', $branchId)
                ->with(['product.brand', 'product.category'])
                ->get();

            return response()->json([
                'success' => true,
                'products' => $productStocks->map(function($productStock) {
                    return [
                        'id' => $productStock->product->id,
                        'name' => $productStock->product->name,
                        'unit' => $productStock->product->unit,
                        'brand_name' => $productStock->product->brand->name ?? '-',
                        'stock' => $productStock->quantity ?? 0,
                    ];
                })
            ]);
        }
    }

    /**
     * Find product by barcode for QR code scanning
     */
    public function findProductByBarcode(Request $request)
    {
        $barcode = $request->input('barcode');
        $branchId = $request->input('branch_id');

        if (!$barcode) {
            return response()->json([
                'success' => false,
                'message' => 'Barcode harus diisi'
            ], 400);
        }

        if (!$branchId) {
            return response()->json([
                'success' => false,
                'message' => 'Silakan pilih cabang terlebih dahulu'
            ], 400);
        }

        // Find product by barcode
        $product = Product::with(['brand', 'category'])
            ->where('barcode', $barcode)
            ->first();

        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Produk dengan barcode "' . $barcode . '" tidak ditemukan!'
            ], 404);
        }

        // Check if product has stock in selected branch
        $productStock = ProductStock::where('product_id', $product->id)
            ->where('branch_id', $branchId)
            ->first();

        if (!$productStock) {
            return response()->json([
                'success' => false,
                'message' => 'Produk "' . $product->name . '" tidak tersedia di cabang ini!'
            ], 404);
        }

        // Get the latest price
        $productPrice = \App\Models\ProductPrice::where('product_id', $product->id)
            ->where('branch_id', $branchId)
            ->orderBy('effective_date', 'desc')
            ->first();

        $purchasePrice = $productPrice ? $productPrice->purchase_price : 0;

        return response()->json([
            'success' => true,
            'product' => [
                'id' => $product->id,
                'name' => $product->name,
                'unit' => $product->unit,
                'brand_name' => $product->brand->name ?? '-',
                'stock' => $productStock->quantity ?? 0,
                'purchase_price' => $purchasePrice,
            ]
        ]);
    }

    /**
     * Find purchase by barcode (for QR code scanning in index page)
     */
    public function findPurchaseByBarcode(Request $request)
    {
        $barcode = $request->input('barcode');

        if (!$barcode) {
            return response()->json([
                'success' => false,
                'message' => 'Barcode harus diisi'
            ], 400);
        }

        // Find purchase by barcode
        $purchase = Purchase::with(['supplier', 'branch', 'user'])
            ->where('barcode', $barcode)
            ->first();

        if (!$purchase) {
            return response()->json([
                'success' => false,
                'message' => 'Pembelian dengan barcode "' . $barcode . '" tidak ditemukan!'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'purchase' => [
                'id' => $purchase->id,
                'purchase_number' => $purchase->purchase_number,
                'barcode' => $purchase->barcode,
                'supplier_name' => $purchase->supplier->name ?? '-',
                'branch_name' => $purchase->branch->name ?? '-',
                'purchase_date' => $purchase->purchase_date->format('d-m-Y'),
                'total_amount' => $purchase->total_amount,
                'status' => $purchase->status,
                'user_name' => $purchase->user->name ?? '-',
            ]
        ]);
    }

    /**
     * Generate unique purchase number
     * Format: PO + YYYYMMDD + URUT(4 digit)
     * Example: PO202512190001
     */
    private function generatePurchaseNumber()
    {
        // Prefix tetap
        $prefix = 'PO';

        // Get date format YYYYMMDD
        $date = date('Ymd');

        // Get last purchase number on TODAY
        $lastPurchase = Purchase::whereDate('created_at', today())
            ->orderBy('id', 'desc')
            ->first();

        if ($lastPurchase) {
            // Extract last 4 digits from purchase_number
            $lastNumber = intval(substr($lastPurchase->purchase_number, -4));
            $newNumber = $lastNumber + 1;
        } else {
            // First purchase today
            $newNumber = 1;
        }

        // Format: PO + YYYYMMDD + 4 digit number
        return $prefix . $date . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate unique barcode for purchase
     */
    private function generateBarcode()
    {
        $length = 100; // total panjang karakter random (tanpa PUR + date)
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*()_-+=[]{}|;:,.<>?';
        $charactersLength = strlen($characters);

        do {
            $randomString = '';
            for ($i = 0; $i < $length; $i++) {
                $randomString .= $characters[random_int(0, $charactersLength - 1)];
            }

            // contoh hasil: PUR250103 + 100 random characters
            $barcode = 'PUR' . date('ymd') . $randomString;

        } while (Purchase::where('barcode', $barcode)->exists());

        return $barcode;
    }

    /**
     * Generate QR Code for purchase
     */
    private function generateQrCode(Purchase $purchase)
    {
        try {
            // Create QR code folder if not exists
            $qrPath = public_path('qrcodes/purchases');
            if (!file_exists($qrPath)) {
                mkdir($qrPath, 0755, true);
            }

            // Generate QR Code using BaconQrCode v3 with SVG (for display)
            $svgRenderer = new ImageRenderer(
                new RendererStyle(300, 10),
                new SvgImageBackEnd()
            );

            $svgWriter = new Writer($svgRenderer);

            // Save QR Code as SVG
            $svgFileName = 'purchase_' . $purchase->id . '.svg';
            $svgFilePath = $qrPath . '/' . $svgFileName;

            $svgWriter->writeFile($purchase->barcode, $svgFilePath);

            // Generate PNG version for download using QR code matrix
            $pngFileName = 'purchase_' . $purchase->id . '.png';
            $pngFilePath = $qrPath . '/' . $pngFileName;

            // Encode data to get QR matrix
            $qrCode = Encoder::encode($purchase->barcode, ErrorCorrectionLevel::L(), 'UTF-8');
            $matrix = $qrCode->getMatrix();

            $matrixWidth = $matrix->getWidth();
            $matrixHeight = $matrix->getHeight();

            // Set scale for better quality (each module = 10 pixels)
            $scale = 10;
            $margin = 20; // margin in pixels

            $imageWidth = $matrixWidth * $scale + (2 * $margin);
            $imageHeight = $matrixHeight * $scale + (2 * $margin);

            // Create PNG image
            $img = imagecreatetruecolor($imageWidth, $imageHeight);
            $white = imagecolorallocate($img, 255, 255, 255);
            $black = imagecolorallocate($img, 0, 0, 0);
            imagefill($img, 0, 0, $white);

            // Draw QR code modules
            for ($y = 0; $y < $matrixHeight; $y++) {
                for ($x = 0; $x < $matrixWidth; $x++) {
                    if ($matrix->get($x, $y) == 1) {
                        imagefilledrectangle(
                            $img,
                            $margin + ($x * $scale),
                            $margin + ($y * $scale),
                            $margin + (($x + 1) * $scale) - 1,
                            $margin + (($y + 1) * $scale) - 1,
                            $black
                        );
                    }
                }
            }

            // Save as PNG
            imagepng($img, $pngFilePath, 0); // 0 = no compression for best quality
            imagedestroy($img);

            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to generate QR Code for purchase ' . $purchase->id . ': ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Download QR Code as JPG
     */
    public function downloadQrCodeJpg(Purchase $purchase)
    {
        try {
            // Generate QR Code matrix using BaconQrCode
            $qrCode = \BaconQrCode\Encoder\Encoder::encode(
                $purchase->barcode,
                \BaconQrCode\Common\ErrorCorrectionLevel::L(),
                'UTF-8'
            );
            $matrix = $qrCode->getMatrix();

            $moduleSize = 15;
            $quietZone = 4;
            $matrixWidth = $matrix->getWidth();

            $imageWidth = ($matrixWidth + ($quietZone * 2)) * $moduleSize;

            // Create image with white background
            $img = imagecreatetruecolor($imageWidth, $imageWidth);
            $white = imagecolorallocate($img, 255, 255, 255);
            $black = imagecolorallocate($img, 0, 0, 0);

            // Fill background
            imagefill($img, 0, 0, $white);

            // Draw QR code modules
            for ($y = 0; $y < $matrixWidth; $y++) {
                for ($x = 0; $x < $matrixWidth; $x++) {
                    if ($matrix->get($x, $y) === 1) {
                        $xPos = ($x + $quietZone) * $moduleSize;
                        $yPos = ($y + $quietZone) * $moduleSize;
                        imagefilledrectangle($img, $xPos, $yPos, $xPos + $moduleSize - 1, $yPos + $moduleSize - 1, $black);
                    }
                }
            }

            // Save to temporary file
            $tempJpgPath = sys_get_temp_dir() . '/qr_purchase_' . $purchase->id . '_' . time() . '.jpg';
            imagejpeg($img, $tempJpgPath, 95);
            imagedestroy($img);

            // Download file and delete after
            $fileName = 'QRCode-Purchase-' . $purchase->purchase_number . '.jpg';

            return response()->download($tempJpgPath, $fileName, [
                'Content-Type' => 'image/jpeg'
            ])->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            abort(500, 'Gagal generate QR Code JPG: ' . $e->getMessage());
        }
    }

    /**
     * Export purchase invoice to PDF
     */
    public function exportInvoicePdf(Purchase $purchase)
    {
        $purchase->load(['supplier', 'branch', 'user', 'purchaseDetails.product']);

        $pdf = Pdf::loadView('transaction.purchase.invoice-pdf', compact('purchase'));
        $pdf->setPaper('a4', 'portrait');

        return $pdf->stream('Faktur-Pembelian-' . $purchase->purchase_number . '.pdf');
    }
}
